﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
namespace cn.m4399.operateBz
{
    public class CloudArchive
    {
        private const string CLOUD_ARCHIVE_JAVA_CLASS_NAME = "cn.m4399.operateBZ.CloudArchive";

        /// <summary>
        /// 是否已初始化
        /// </summary>
        /// <returns></returns>
        public static bool IsInited()
        {
            using (var javaClass = new AndroidJavaClass(CLOUD_ARCHIVE_JAVA_CLASS_NAME))
            {
                return javaClass.CallStatic<bool>("isInited");
            }
        }

        /// <summary>
        /// 初始化
        /// </summary>
        /// <param name="publicKey"></param>
        /// <param name="initListener"></param>
        public static void Init(string publicKey, InitListener initListener)
        {
            using (var javaClass = new AndroidJavaClass(CLOUD_ARCHIVE_JAVA_CLASS_NAME))
            {
                javaClass.CallStatic("init", publicKey, initListener);
            }
        }

        /// <summary>
        /// 获取当前云存档接口对象，请不要保存返回的对象
        /// </summary>
        /// <returns></returns>
        public static CloudArchive GetCloudArchive()
        {
            using (var javaClass = new AndroidJavaClass(CLOUD_ARCHIVE_JAVA_CLASS_NAME))
            {
                var cloudArchiveJavaObject = javaClass.CallStatic<AndroidJavaObject>("getCloudArchive");
                return new CloudArchive(cloudArchiveJavaObject);
            }
        }

        public abstract class InitListener : AndroidJavaProxy
        {
            public InitListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$InitListener")
            {

            }

            internal void onSuccess()
            {
                OnSuccess();
            }

            internal void onFailure(int code, string message)
            {
                OnFailure(code, message);
            }

            internal void onUserChanged(AndroidJavaObject previousUserCloudArchiveJavaObject, AndroidJavaObject currentUserCloudArchiveJavaObject)
            {
                OnUserChanged(previousUserCloudArchiveJavaObject != null ? new CloudArchive(previousUserCloudArchiveJavaObject) : null, 
                    currentUserCloudArchiveJavaObject != null ? new CloudArchive(currentUserCloudArchiveJavaObject) : null);
            }


            /// <summary>
            /// 初始化成功
            /// </summary>
            protected abstract void OnSuccess();

            /// <summary>
            /// 初始化故障
            /// </summary>
            /// <param name="code"></param>
            /// <param name="message"></param>
            protected abstract void OnFailure(int code, string message);

            /// <summary>
            /// 登录的用户改变
            /// </summary>
            /// <param name="previousUserCloudArchive">nullable</param>
            /// <param name="currentUserCloudArchive">nullable</param>
            protected abstract void OnUserChanged(CloudArchive previousUserCloudArchive, CloudArchive currentUserCloudArchive);
        }

        public abstract class ListListener : AndroidJavaProxy
        {
            public ListListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$ListListener")
            {
            }

            internal void onSuccess(AndroidJavaObject archiveListJavaObject)
            {
                var archiveList = new List<Archive>();
                var size = archiveListJavaObject.Call<int>("size");
                for (var i = 0; i < size; ++ i)
                {
                    var archiveJavaObject = archiveListJavaObject.Call<AndroidJavaObject>("get", i);
                    var archive = new Archive(archiveJavaObject);
                    archiveList.Add(archive);
                }
                OnSuccess(archiveList);
            }

            internal void onFailure(int code, string message)
            {
                OnFailure(code, message);
            }

            /// <summary>
            /// 获取存档列表成功
            /// </summary>
            /// <param name="archiveList"></param>
            protected abstract void OnSuccess(List<Archive> archiveList);

            /// <summary>
            /// 获取存档列表故障
            /// </summary>
            /// <param name="code"></param>
            /// <param name="message"></param>
            protected abstract void OnFailure(int code, string message);
        }

        public abstract class GetListener : AndroidJavaProxy
        {
            public GetListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$GetListener")
            {
            }

            internal void onSuccess(AndroidJavaObject archiveJavaObject)
            {
                var archive = new Archive(archiveJavaObject);
                OnSuccess(archive);
            }

            internal void onFailure(int code, string message)
            {
                OnFailure(code, message);
            }

            /// <summary>
            /// 读取存档成功
            /// </summary>
            /// <param name="archive"></param>
            protected abstract void OnSuccess(Archive archive);

            /// <summary>
            /// 读取存档故障
            /// </summary>
            /// <param name="code"></param>
            /// <param name="message"></param>
            protected abstract void OnFailure(int code, string message);
        }

        public abstract class SaveListener : AndroidJavaProxy
        {
            public SaveListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$SaveListener")
            {
            }

            internal void onSuccess()
            {
                OnSuccess();
            }

            internal void onFailure(int code, string message)
            {
                OnFailure(code, message);
            }

            /// <summary>
            /// 保存存档成功
            /// </summary>
            protected abstract void OnSuccess();

            /// <summary>
            /// 保存存档故障
            /// </summary>
            /// <param name="code"></param>
            /// <param name="message"></param>
            protected abstract void OnFailure(int code, string message);
        }


        private readonly AndroidJavaObject JavaObject;

        CloudArchive(AndroidJavaObject javaObject)
        {
            JavaObject = javaObject;
        }

        /// <summary>
        /// 存档列表
        /// </summary>
        /// <param name="listListener"></param>
        public void List(ListListener listListener)
        {
            JavaObject.Call("list", listListener);
        }

        /// <summary>
        /// 保存存档
        /// </summary>
        /// <param name="archive"></param>
        /// <param name="saveListener"></param>
        public void Save(Archive archive, SaveListener saveListener)
        {
            JavaObject.Call("save", archive.GetJavaObject(), saveListener);
        }

        /// <summary>
        /// 读取存档
        /// </summary>
        /// <param name="index"></param>
        /// <param name="getListener"></param>
        public void Get(int index, GetListener getListener)
        {
            JavaObject.Call("get", index, getListener);
        }

        /// <summary>
        /// 云存档读档监听器
        /// </summary>
        public abstract class OnSimpleStrGetListener : AndroidJavaProxy
        {
            public OnSimpleStrGetListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$SimpleStrGetListener")
            {

            }

            internal void onSuccess(string arch)
            {
                OnSuccess(arch);
            }

            internal void onFailure(int resultCode, string msg)
            {
                OnFailure(resultCode, msg);
            }

            /// <summary>
            /// 读档成功
            /// </summary>
            /// <param name="arch"></param>
            protected abstract void OnSuccess(string arch);

            /// <summary>
            /// 读档失败
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            protected abstract void OnFailure(int resultCode, string msg);

        }

        /// <summary>
        /// 云存档上传监听器
        /// </summary>
        public abstract class OnSimpleStrSetListener : AndroidJavaProxy
        {
            public OnSimpleStrSetListener() : base(CLOUD_ARCHIVE_JAVA_CLASS_NAME + "$SimpleStrSetListener")
            {

            }

            internal void onSuccess()
            {
                OnSuccess();
            }

            internal void onFailure(int resultCode, string msg)
            {
                OnFailure(resultCode, msg);
            }

            /// <summary>
            /// 上传成功
            /// </summary>
            protected abstract void OnSuccess();

            /// <summary>
            /// 上传失败
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            protected abstract void OnFailure(int resultCode, string msg);

        }
    }

}