package cn.m4399.rpgsample;

import android.annotation.SuppressLint;
import android.content.pm.ActivityInfo;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.webkit.JavascriptInterface;
import android.webkit.WebSettings;
import android.webkit.WebViewClient;
import android.widget.Toast;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import com.example.rpgsample.R;
import com.example.rpgsample.databinding.ActivityMainBinding;

import org.json.JSONException;
import org.json.JSONObject;

import cn.m4399.operateBZ.CloudArchive;
import cn.m4399.operateBZ.OperateCenter;
import cn.m4399.operateBZ.OperateConfig;
import cn.m4399.operateBZ.User;

public class RPGMakerActivity extends AppCompatActivity {
    private static final String TAG = "【OperateBzDemo】";
    private ActivityMainBinding mActivityMainBinding;
    private final RechargeHelper rechargeHelper = new RechargeHelper();

    /**
     * game key
     */
    private static final String GAME_KEY = "40027";

    /**
     * 屏幕方向
     */
    private static final int GAME_ORIENTATION = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;

    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);
        if (hasFocus) {
            int flags = View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                    | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                    | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                    | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                    | View.SYSTEM_UI_FLAG_FULLSCREEN;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
                flags |= View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY;
            getWindow().getDecorView().setSystemUiVisibility(flags);
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mActivityMainBinding = ActivityMainBinding.inflate(getLayoutInflater());
        setContentView(mActivityMainBinding.getRoot());
        loadGame();
        initSDK();
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();
    }

    @SuppressLint("SetJavaScriptEnabled")
    private void gameWebViewConf() {
        WebSettings webSettings = mActivityMainBinding.webView.getSettings();
        mActivityMainBinding.webView.setWebViewClient(new WebViewClient());
        webSettings.setJavaScriptEnabled(true);
        webSettings.setAllowFileAccessFromFileURLs(true);
        webSettings.setDomStorageEnabled(true);
    }

    @SuppressLint("AddJavascriptInterface")
    private void loadGame() {
        gameWebViewConf();
        mActivityMainBinding.webView.loadUrl("file:///android_asset/ext/index.html"); // 游戏接入时应改成对应的index目录
        mActivityMainBinding.webView.addJavascriptInterface(new OperateJsInterface(), "opeNativeApi");
    }

    private void initSDK() {
        // 配置全局属性，如横竖屏配置，完整的详细说明见接入文档
        OperateCenter mOpeCenter = OperateCenter.getInstance();
        OperateConfig opeConfig = new OperateConfig.Builder(this)
                .setDebugEnabled(false)
                .setOrientation(GAME_ORIENTATION)
                .setGameKey(GAME_KEY)
                .build();

        mOpeCenter.setConfig(opeConfig);
        mOpeCenter.init(this, new OperateCenter.OnInitGlobalListener() {
            @Override
            public void onInitFinished(boolean isLogin, User user) {
                String state = isLogin ? getString(R.string.m4399_ope_bz_login_already) : getString(R.string.m4399_ope_bz_login_not_yet);
                toastAndLog(R.string.m4399_ope_bz_init_result, state, user);
                invokeJsMethod("init_callback", isLogin, state, userToJson(user));
            }

            @Override
            public void onUserAccountLogout(boolean fromUserCenter) {
                toastAndLog(R.string.m4399_ope_bz_main_label_quit);
                invokeJsMethod("exitAccount_callback", "", getString(R.string.m4399_ope_bz_main_label_quit));
            }

            @Override
            public void onSwitchUserAccountFinished(boolean fromUserCenter, User user) {
                //无需处理
            }
        });
    }

    @Keep
    private final class OperateJsInterface {
        @JavascriptInterface
        public void login(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    OperateCenter.getInstance().login(RPGMakerActivity.this, new OperateCenter.OnLoginFinishedListener() {
                        @Override
                        public void onLoginFinished(boolean success, int resultCode, User user) {
                            String msg = success ? getString(R.string.m4399_ope_bz_login_success) :
                                    getString(R.string.m4399_ope_bz_login_failed);
                            toastAndLog(R.string.m4399_ope_bz_login_result, msg, resultCode, user);
                            invokeJsMethod(cbName, resultCode, msg, userToJson(user));
                        }
                    });
                }
            });
        }

        @JavascriptInterface
        public void recharge(String value, String mark, String commodity, String cbName) {
            rechargeHelper.recharge(RPGMakerActivity.this, value, mark, commodity, new OperateCenter.OnRechargeFinishedListener() {
                @Override
                public void onRechargeFinished(boolean success, int resultCode, String msg) {
                    toastAndLog(getString(R.string.m4399_ope_bz_recharge_result, resultCode, msg));
                    invokeJsMethod(cbName, resultCode, msg);
                }
            });
        }

        @JavascriptInterface
        public void getArchive(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    OperateCenter.getInstance().getArchiveStringSimple(new CloudArchive.SimpleStrGetListener() {
                        @Override
                        public void onSuccess(@NonNull String archive) {
                            invokeJsMethod(cbName, archive);
                            toastAndLog(getString(R.string.m4399_ope_bz_archive_get_success));
                        }

                        @Override
                        public void onFailure(int code, @NonNull String message) {
                            invokeJsMethod(cbName, String.valueOf(code), message);
                            toastAndLog(getString(R.string.m4399_ope_bz_archive_get_failed_tip));
                        }
                    });
                }
            });
        }

        @JavascriptInterface
        public void setArchive(String value, String cbName) {
            if (TextUtils.isEmpty(value)) {
                value = "4399test";
            }
            String archiveData = value;
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    OperateCenter.getInstance().setArchiveStringSimple(archiveData, new CloudArchive.SimpleStrSetListener() {
                        @Override
                        public void onSuccess() {
                            invokeJsMethod(cbName, "", getString(R.string.m4399_ope_bz_archive_get_success));
                            toastAndLog(getString(R.string.m4399_ope_bz_archive_set_success));
                        }

                        @Override
                        public void onFailure(int code, @NonNull String message) {
                            invokeJsMethod(cbName, String.valueOf(code), message);
                            toastAndLog(getString(R.string.m4399_ope_bz_archive_set_failed_tip));
                        }
                    });
                }
            });
        }

        @JavascriptInterface
        public void exitAccount(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    OperateCenter.getInstance().logout();
                }
            });
        }

        @JavascriptInterface
        public void getCurrentUser(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    invokeJsMethod(cbName, String.valueOf(userToJson(OperateCenter.getInstance().getCurrentAccount())));
                }
            });
        }

        @JavascriptInterface
        public void getIsLogin(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    invokeJsMethod(cbName, getString(OperateCenter.getInstance().isLogin() ? R.string.login_already : R.string.login_not_yet));
                }
            });
        }

        @JavascriptInterface
        public void getSdkVersion(String cbName) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    invokeJsMethod(cbName, getString(R.string.demo_and_sdk_info, OperateCenter.getVersion(), GAME_KEY));
                }
            });
        }
    }

    private String userToJson(User user) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("name", user.getName());
            jsonObject.put("uid", user.getUid());
            jsonObject.put("state", user.getState());
            jsonObject.put("phone", user.getPhone());
            jsonObject.put("idCardState", user.getIdCardState());
            jsonObject.put("nick", user.getNick());
            jsonObject.put("isIdCheckedReal", user.isIdCheckedReal());
            jsonObject.put("isIdCardEditable", user.isIdCardEditable());
            return jsonObject.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return "{}";
    }

    private void invokeJsMethod(String methodName, Object... args) {
        StringBuilder argv = new StringBuilder();
        String protocol;
        if (args != null && args.length > 0) {
            for (Object o : args) {
                Object v = o;
                if (o == null)
                    v = "undefined";
                else if (o instanceof String)
                    v = "'" + o + "'";

                if (argv.length() > 0)
                    argv.append(",").append(v);
                else
                    argv.append(v);
            }
            protocol = String.format("javascript:window.%s(%s);", methodName, argv);
        } else {
            protocol = String.format("javascript:window.%s();", methodName);
        }
        mActivityMainBinding.webView.loadUrl(protocol);
    }

    protected void toastAndLog(int formatter, Object... args) {
        toastAndLog(getString(formatter, args));
    }

    public void toastAndLog(String msg) {
        Toast.makeText(this, TAG + msg, Toast.LENGTH_SHORT).show();
        Log.d(TAG, msg);
    }
}