package com.operatebz.js;

import android.app.Activity;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.JavascriptInterface;
import android.webkit.ValueCallback;
import android.webkit.WebView;

import androidx.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

import cn.m4399.operateBZ.CloudArchive;
import cn.m4399.operateBZ.OperateCenter;
import cn.m4399.operateBZ.OperateConfig;
import cn.m4399.operateBZ.Order;
import cn.m4399.operateBZ.User;

public class OperateBZJsApi {
  public static final String TAG = "【OperateBZJsApi】";
  private final Activity mActivity;
  private final WebView mWebView;
  private boolean mHadInit;

  public OperateBZJsApi(Activity activity, WebView webView) {
    mActivity = activity;
    mWebView = webView;
  }
  
  private String getString(int resId) {
    return mActivity.getString(resId);
  }

  @JavascriptInterface
  public void initSDK(String gameId, int screenOrientation, boolean compatNotch, String cbName, String cbOnLogout) {
    Log.v(TAG, String.format("initSDK: %s, %s, %s, %s, %s,",
      gameId, screenOrientation, compatNotch, cbName, cbOnLogout));
    mActivity.runOnUiThread(new Runnable() {
      @Override
      public void run() {
        if (mHadInit) {
          invokeJsMethod(cbName, String.valueOf(0), getString(R.string.m4399_ope_bz_js_init_once_more), 
            userToJson(OperateCenter.getInstance().getCurrentAccount()));
        } else {
          // 配置全局属性，如横竖屏配置，完整的详细说明见接入文档
          OperateCenter mOpeCenter = OperateCenter.getInstance();
          OperateConfig opeConfig = new OperateConfig.Builder(mActivity)
            .setDebugEnabled(false)
            .setOrientation(screenOrientation)
            .setGameKey(gameId)
            .build();

          mOpeCenter.setConfig(opeConfig);
          mOpeCenter.init(mActivity, new OperateCenter.OnInitGlobalListener() {
            @Override
            public void onInitFinished(boolean isLogin, User user) {
              Log.v(TAG, String.format("initSDK  onResult : %s, %s, %s,",
                isLogin, userToJson(user), cbName));
              invokeJsMethod(cbName, isLogin, userToJson(user));
              mHadInit = true;
            }

            @Override
            public void onUserAccountLogout(boolean fromUserCenter) {
              Log.v(TAG, String.format("onLogout  : %s", cbOnLogout));
              invokeJsMethod(cbOnLogout, 0, String.format(" onLogout : %s,", cbOnLogout));
            }

            @Override
            public void onSwitchUserAccountFinished(boolean fromUserCenter, User user) {
              //无需处理
            }
          });
        }
      }
    });
  }

  @JavascriptInterface
  public void login(final String cbName) {
    Log.v(TAG, String.format("login  : %ss,", cbName));
    mActivity.runOnUiThread(new Runnable() {
      @Override
      public void run() {
        OperateCenter.getInstance().login(mActivity, (success, code, user) -> {
          Log.v(TAG, String.format("login  onResult : %s, %s, %s, %s,",
            success, code, userToJson(user), cbName));
          if (!TextUtils.isEmpty(cbName)) {
            invokeJsMethod(cbName, code, userToJson(user));
          }
        });
      }
    });
  }

  @JavascriptInterface
  public void getArchive(String cbName) {
    mActivity.runOnUiThread(new Runnable() {
      @Override
      public void run() {
        OperateCenter.getInstance().getArchiveStringSimple(new CloudArchive.SimpleStrGetListener() {
          @Override
          public void onSuccess( @NonNull String archive) {
            if (!TextUtils.isEmpty(cbName)) {
              invokeJsMethod(cbName, archive);
            }
          }

          @Override
          public void onFailure(int code,  @NonNull String message) {
            if (!TextUtils.isEmpty(cbName)) {
              invokeJsMethod(cbName, String.valueOf(code), message);
            }
          }
        });
      }
    });
  }

  @JavascriptInterface
  public void setArchive(String value, String cbName) {
    if (TextUtils.isEmpty(value)) {
      if (!TextUtils.isEmpty(cbName))
        invokeJsMethod(cbName, 3, getString(R.string.m4399_ope_bz_js_archive_date_empty));
      return;
    }
    mActivity.runOnUiThread(new Runnable() {
      @Override
      public void run() {
        OperateCenter.getInstance().setArchiveStringSimple(value, new CloudArchive.SimpleStrSetListener() {
          @Override
          public void onSuccess() {
            if (!TextUtils.isEmpty(cbName))
              invokeJsMethod(cbName, 0, getString(R.string.m4399_ope_bz_js_archive_success));
          }

          @Override
          public void onFailure(int code,  @NonNull String message) {
            if (!TextUtils.isEmpty(cbName))
              invokeJsMethod(cbName, String.valueOf(code), message);
          }
        });
      }
    });
  }

  @JavascriptInterface
  public void logout() {
    mActivity.runOnUiThread(() -> OperateCenter.getInstance().logout());
  }

  @JavascriptInterface
  public String getUser() {
    return userToJson(OperateCenter.getInstance().getCurrentAccount());
  }

  @JavascriptInterface
  public boolean isLogin() {
    return OperateCenter.getInstance().isLogin();
  }

  @JavascriptInterface
  public String getSdkVersion() {
    return OperateCenter.getVersion();
  }

  @JavascriptInterface
  public void recharge(String value, String mark, String commodity, String cbName) {
    Log.v(TAG, String.format("recharge : %s, %s, %s, %s,",
      value, mark, commodity, cbName));
    mActivity.runOnUiThread(new Runnable() {
      @Override
      public void run() {
        int money = 6;
        if (!TextUtils.isEmpty(value) && Integer.parseInt(value) > 0) {
          money = Integer.parseInt(value);
        }
        // C：充值
        if (TextUtils.isEmpty(mark)) {
          if (cbName != null) {
            invokeJsMethod(cbName, 3, getString(R.string.m4399_ope_bz_js_api_recharge_mark_empty));
          }
          return;
        }
        String commodityName = commodity;
        if (TextUtils.isEmpty(commodityName)) {
          commodityName = mActivity.getString(R.string.m4399_ope_bz_js_api_commodity, money);
        }
        OperateCenter.getInstance().recharge(mActivity,
          // 充值金额，整数，单位元
          new Order(money, mark)
            // 是否支持超出金额，默认不支持
            .supportExcess(false)
            // 商品名，可选，不传时认为商品名是游戏币
            .commodity(commodityName),
          new OperateCenter.OnRechargeFinishedListener() {
            @Override
            public void onRechargeFinished(boolean success, int resultCode, String msg) {
              Log.v(TAG, String.format("recharge  onResult : %s, %s, %s, %s,",
                success, resultCode, msg, cbName));
              // 处理充值结果
              if (cbName != null) {
                invokeJsMethod(cbName, resultCode, msg);
              }
            }
          });
      }
    });
  }

  private String userToJson(User user) {
    try {
      JSONObject jsonObject = new JSONObject();
      jsonObject.put("name", user.getName());
      jsonObject.put("uid", user.getUid());
      jsonObject.put("state", user.getState());
      jsonObject.put("phone", user.getPhone());
      jsonObject.put("idCardState", user.getIdCardState());
      jsonObject.put("nick", user.getNick());
      return jsonObject.toString();
    } catch (JSONException e) {
      e.printStackTrace();
    }
    return "{}";
  }

  /*
   * @param methodName js方法名称
   * @param args参数列表，复杂对象可转为 json 字符串，在 js 处再解析
   */
  private void invokeJsMethod(String methodName, Object... args) {
    StringBuilder argv = new StringBuilder();
    String protocol;
    if (args != null && args.length > 0) {
      for (Object o : args) {
        Object v = o;
        if (o == null)
          v = "undefined";
        else if (o instanceof String)
          v = "'" + o + "'";

        if (argv.length() > 0)
          argv.append(",").append(v);
        else
          argv.append(v);
      }
      protocol = String.format("javascript:window.%s(%s);", methodName, argv);
    } else {
      protocol = String.format("javascript:window.%s();", methodName);
    }
    int n = protocol.length();
    Log.v(TAG, String.format("invoke js method: %s",
      n > 500 ? protocol.substring(0, 300) + "..." + protocol.substring(n - 200) : protocol));

    if (mWebView != null) {
      mActivity.runOnUiThread(new Runnable() {
        @Override
        public void run() {
          mWebView.evaluateJavascript(protocol, new ValueCallback<>() {
            @Override
            public void onReceiveValue(String s) {
              
            }
          });
        }
      });
    }
  }
}
