﻿using UnityEngine;
using System.Collections.Generic;

namespace cn.m4399.operate.api
{
    /// <summary>
    /// GameChart
    /// 4399 sdk
    /// </summary>
    public class GameChart
    {
        private const string OPERATE_GAME_CHART_JAVA_CLASS_NAME = "cn.m4399.operate.api.GameChart";

        /// <summary>
        /// 模块初始化
        /// </summary>
        /// <param name="publicKey">公钥</param>
        /// <returns></returns>
        public static void Init(string publicKey)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("init", publicKey);
            }
        }

        /// <summary>
        /// 获取当前用户头像地址、昵称
        /// </summary>
        /// <param name="serverId">区服id，未分服传0</param>
        /// <param name="roleId">游戏角色id，没有可传0</param>
        /// <param name="listener">回调对象</param>
        public static void getPtUser(OpeDataPtUserListener listener)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("getPtUser", listener);
            }
        }

        /// <summary>
        /// 获取角色排名数据
        /// </summary>
        /// <param name="rankId">榜单id，必须</param>
        /// <param name="serverId">区服id，未分服传0</param>
        /// <param name="roleId">游戏角色id，没有可传0</param>
        /// <param name="before">当前角色前几名</param>
        /// <param name="after">当前角色后几名</param>
        /// <param name="listener">回调对象</param>
        public static void GetChartAroundRole(string rankId, int serverId, string roleId, int before, int after, OpeDataChartItemListListener listener)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("getChartAroundRole", rankId, serverId, roleId, before, after, listener);
            }
        }

        /// <summary>
        /// 获取榜单排名区间数据
        /// </summary>
        /// <param name="rankId">榜单id，必须</param>
        /// <param name="serverId">区服id，未分服传0</param>
        /// <param name="start">榜单排名区间起始index</param>
        /// <param name="end">榜单排名区间结束index</param>
        /// <param name="listener">回调对象</param>
        public static void GetChartByRange(string rankId, int serverId, int start, int end, OpeDataChartItemListListener listener)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("getChartByRange", rankId, serverId, start, end, listener);
            }
        }

        /// <summary>
        /// 更新角色排行分数
        /// </summary>
        /// <param name="rankId">榜单id，必须</param>
        /// <param name="serverId">区服id，未分服传0</param>
        /// <param name="roleId">游戏角色id，没有可传0</param>
        /// <param name="roleAvatar">头像（无头像时，可null）</param>
        /// <param name="roleName">角色名/昵称（无角色名/昵称时，可null）</param>
        /// <param name="score">排名数值</param>
        /// <param name="listener">回调对象</param>
        public static void UpdateByRole(string rankId, int serverId, string roleId, string roleAvatar, string roleName, float score, OpeDataUpdateChartListener listener)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("updateByRole", rankId, serverId, roleId, roleAvatar, roleName, score, listener);
            }
        }

        /// <summary>
        /// 获取所有榜单数据
        /// </summary>
        /// <param name="listener">回调对象</param>
        public static void GetGameChart(OpeDataChartDescriptorListListener listener)
        {
            using (var gameChartClass = new AndroidJavaClass(OPERATE_GAME_CHART_JAVA_CLASS_NAME))
            {
                gameChartClass.CallStatic("getGameChart", listener);
            }
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataPtUserListener : AndroidJavaProxy
        {
            public OpeDataPtUserListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject)
            {
                OnFinished(resultCode, msg, androidJavaObject);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="androidJavaObject"></param>
            protected abstract void OnFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject);
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataUpdateChartListener : AndroidJavaProxy
        {
            public OpeDataUpdateChartListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, bool success)
            {
                OnFinished(resultCode, msg, success);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="androidJavaObject"></param>
            protected abstract void OnFinished(int resultCode, string msg, bool success);
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataChartItemListListener : AndroidJavaProxy
        {
            public OpeDataChartItemListListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject chartItemListJavaObject)
            {
                var chartItemList = new List<ChartItem>();
                var size = chartItemListJavaObject.Call<int>("size");
                for (var i = 0; i < size; ++ i)
                {
                    var chartItemJavaObject = chartItemListJavaObject.Call<AndroidJavaObject>("get", i);
                    var chartItem = new ChartItem(chartItemJavaObject);
                    chartItemList.Add(chartItem);
                }
                OnFinished(resultCode, msg, chartItemList);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="chartItemList"></param>
            protected abstract void OnFinished(int resultCode, string msg, List<ChartItem> chartItemList);
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataChartDescriptorListListener : AndroidJavaProxy
        {
            public OpeDataChartDescriptorListListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject chartDescriptorListJavaObject)
            {
                var chartDescriptorList = new List<ChartDescriptor>();
                var size = chartDescriptorListJavaObject.Call<int>("size");
                for (var i = 0; i < size; ++ i)
                {
                    var chartDescriptorJavaObject = chartDescriptorListJavaObject.Call<AndroidJavaObject>("get", i);
                    var chartDescriptor = new ChartDescriptor(chartDescriptorJavaObject);
                    chartDescriptorList.Add(chartDescriptor);
                }
                OnFinished(resultCode, msg, chartDescriptorList);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="chartItemList"></param>
            protected abstract void OnFinished(int resultCode, string msg, List<ChartDescriptor> chartDescriptorList);
        }
    }
}
