﻿using UnityEngine;
using System.Collections.Generic;

namespace cn.m4399.operate.api
{
    /// <summary>
    /// Friendship
    /// 4399 sdk
    /// </summary>
    public class Friendship
    {
        private const string OPERATE_Friendship_JAVA_CLASS_NAME = "cn.m4399.operate.api.Friendship";
        /// 关注列表
        public const int FOLLOW = 0;
        /// 粉丝列表
        public const int FOLLOWER = 1;
        /// 互关列表
        public const int MUTUAL = 2;
        /// 黑名单列表
        public const int BLACK = 3;

        /// <summary>
        /// 获取自身用户信息
        /// </summary>
        /// <param name="listener"></param>
        public static void GetMyself(OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("getMyself", listener);
            }
        }

        /// <summary>
        /// 获取好友关系列表
        /// </summary>
        /// <param name="type">需要获取的某一列表，包含 {@value FOLLOW},{@value FOLLOWER},{@value MUTUAL},{@value BLACK}</param>
        /// <param name="page">分页页码，获取粉丝列表时可分页，默认为 1</param>
        /// <param name="listener"></param>
        public static void List(int type, int page, OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("list", type, page, listener);
            }
        }

        /// <summary>
        /// 当前用户关注其他用户
        /// </summary>
        /// <param name="uids">被关注用户uid,多个以英文逗号隔开</param>
        /// <param name="listener"></param>
        public static void Follow(string uids, OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("follow", uids, listener);
            }
        }

        /// <summary>
        /// 当前用户取消关注其他用户
        /// </summary>
        /// <param name="uid">被取消关注用户uid</param>
        /// <param name="listener"></param>
        public static void Unfollow(string uid, OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("unfollow", uid, listener);
            }
        }

        /// <summary>
        /// 当前用户使用精确uid搜索用户
        /// </summary>
        /// <param name="uid">被搜索用户uid，4399 账号系统中的用户标识</param>
        /// <param name="listener"></param>
        public static void Search(string uid, OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("search", uid, listener);
            }
        }

        /// <summary>
        /// 私信邀请
        /// </summary>
        /// <param name="uid">接收方用户uid</param>
        /// <param name="listener"></param>
        public static void Invite(string uid, OpeDataFriendshipListener listener)
        {
            using (var friendshipClass = new AndroidJavaClass(OPERATE_Friendship_JAVA_CLASS_NAME))
            {
                friendshipClass.CallStatic("invite", UnityAndroid.GetUnityActivity(), uid, listener);
            }
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataFriendshipListener : AndroidJavaProxy
        {
            public OpeDataFriendshipListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject)
            {
                OnFinished(resultCode, msg, androidJavaObject);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="androidJavaObject"></param>
            protected abstract void OnFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject);
        }

    }
}
