﻿using UnityEngine;
using System.Collections.Generic;

namespace cn.m4399.operate.api
{
    /// <summary>
    /// Achievement
    /// 4399 sdk
    /// </summary>
    public class Achievement
    {
        private const string OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME = "cn.m4399.operate.api.Achievement";

        /// <summary>
        /// 模块初始化
        /// </summary>
        /// <param name="publicKey">公钥</param>
        /// <returns></returns>
        public static void Init(string publicKey)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                achievementClass.CallStatic("init", publicKey);
            }
        }

        /// <summary>
        /// 初始化角色信息
        /// </summary>
        /// <param name="serverId">区服id，没有则传字符串0</param>
        /// <param name="roleId">游戏角色id，没有则传字符串0</param>
        /// <param name="listener">回调对象</param>
        public static void InitRole(string serverId, string roleId, OpeDataListener listener)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                achievementClass.CallStatic("initRole", serverId, roleId, listener);
            }
        }

        /// <summary>
        /// 检查角色在成就系统中否初始化
        /// </summary>
        /// <param name="serverId">区服id，没有则传字符串0</param>
        /// <param name="roleId">游戏角色id，没有则传字符串0</param>
        public static bool IsRoleInit(string serverId, string roleId)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                return achievementClass.CallStatic<bool>("isRoleInit", serverId, roleId);
            }
        }

        /// <summary>
        /// 获取角色成就列表
        /// </summary>
        /// <param name="serverId">区服id，没有则传字符串0</param>
        /// <param name="roleId">游戏角色id，没有则传字符串0</param>
        /// <param name="listener">回调对象</param>
        public static void GetAchievementList(string serverId, string roleId, OpeDataListListener listener)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                achievementClass.CallStatic("getAchievementList", serverId, roleId, listener);
            }
        }

        /// <summary>
        /// 更新角色成就
        /// </summary>
        /// <param name="serverId">区服id，没有则传字符串0</param>
        /// <param name="roleId">游戏角色id，没有则传字符串0</param>
        /// <param name="model">更新模式，枚举变量，详情参考 UpdateMode 说明</param>
        /// <param name="id">需要更新的成就id</param>
        /// <param name="step">需要更新的成就步数，一般是正整数。当更新模式巍峨 UpdateMode.UPDATE_MAX时传固定值 -1</param>
        /// <param name="listener">回调对象</param>
        public static void UpdateAchievement(string serverId, string roleId, UpdateMode model, string id, int step, OpeDataListener listener)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                using (var enumJavaClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME + "$UpdateMode")) {
                    var enumJava = enumJavaClass.GetStatic<AndroidJavaObject>(model.ToString());
                    achievementClass.CallStatic("updateAchievement", serverId, roleId, enumJava, id, step, listener);
                }
            }
        }

        /// <summary>
        /// 获取本游戏成就列表
        /// </summary>
        /// <param name="listener">回调对象</param>
        public static void GetGameAchievement(OpeDataListListener listener)
        {
            using (var achievementClass = new AndroidJavaClass(OPERATE_ACHIEVEMENT_JAVA_CLASS_NAME))
            {
                achievementClass.CallStatic("getGameAchievement", listener);
            }
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataListener : AndroidJavaProxy
        {
            public OpeDataListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject)
            {
                OnFinished(resultCode, msg, androidJavaObject);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="androidJavaObject"></param>
            protected abstract void OnFinished(int resultCode, string msg, AndroidJavaObject androidJavaObject);
        }

        /// <summary>
        /// 数据监听器
        /// </summary>
        public abstract class OpeDataListListener : AndroidJavaProxy
        {
            public OpeDataListListener() : base("cn.m4399.operate.api.OpeDataListener")
            {

            }

            internal void onFinished(int resultCode, string msg, AndroidJavaObject achievementListJavaObject)
            {
                var achievementList = new List<AchievementDetail>();
                var size = achievementListJavaObject.Call<int>("size");
                for (var i = 0; i < size; ++ i)
                {
                    var achievementJavaObject = achievementListJavaObject.Call<AndroidJavaObject>("get", i);
                    var achievement = new AchievementDetail(achievementJavaObject);
                    achievementList.Add(achievement);
                }
                OnFinished(resultCode, msg, achievementList);
            }

            /// <summary>
            /// 操作结果返回
            /// </summary>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            /// <param name="achievementList"></param>
            protected abstract void OnFinished(int resultCode, string msg, List<AchievementDetail> achievementList);
        }

        public enum UpdateMode
        {
            /// 累加，将数值加到旧值上
            UPDATE_ADD,
            /// 覆盖，用新值覆盖旧值
            UPDATE_SET,
            /// 设为最大值
            UPDATE_MAX
        }
    }
}
