﻿using UnityEngine;

namespace cn.m4399.operate
{
    /// <summary>
    /// OperateCenter
    /// 4399 sdk
    /// </summary>
    public class OperateCenter
    {
        private const string OPERATE_CENTER_JAVA_CLASS_NAME = "cn.m4399.operate.OperateCenter";

        private readonly AndroidJavaObject JavaObject;

        private OperateCenter(AndroidJavaObject javaObject)
        {
            JavaObject = javaObject;
        }

        /// <summary>
        /// 获取运营SDK实例
        /// </summary>
        /// <returns></returns>
        public static OperateCenter GetInstance()
        {
            using (var operateCenterClass = new AndroidJavaClass(OPERATE_CENTER_JAVA_CLASS_NAME))
            {
                var instance = operateCenterClass.CallStatic<AndroidJavaObject>("getInstance");
                return new OperateCenter(instance);
            }
        }

        /// <summary>
        /// 运营SDK版本号
        /// </summary>
        /// <returns></returns>
        public static string GetVersion()
        {
            using (var operateCenterClass = new AndroidJavaClass(OPERATE_CENTER_JAVA_CLASS_NAME))
            {
                return operateCenterClass.CallStatic<string>("getVersion");
            }
        }

        /// <summary>
        /// 配置SDK
        /// </summary>
        /// <param name="operateConfig"></param>
        public void SetConfig(OperateConfig operateConfig)
        {
            JavaObject.Call("setConfig", operateConfig.GetJavaObject());
        }

        /// <summary>
        /// 游戏方用户协议及隐私政策
        /// </summary>
        /// <param name="OpeResultListener"></param>
        public void gameProtocol(OperateConfig operateConfig, OpeResultListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("gameProtocol", UnityAndroid.GetUnityActivity(), operateConfig.GetJavaObject(),listener);
            });
        }

        /// <summary>
        /// 初始化运营SDK
        /// </summary>
        /// <param name="onInitGlobalListener"></param>
        public void Init(OnInitGlobalListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("init", UnityAndroid.GetUnityActivity(), listener);
            });
        }

        /// <summary>
        /// 登录
        /// </summary>
        /// <param name="listener"></param>
        public void Login(OnLoginFinishedListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("login", UnityAndroid.GetUnityActivity(), listener);
            });
        }

        /// <summary>
        /// 切换用户
        /// </summary>
        /// <param name="listener"></param>
        public void SwitchAccount(OnLoginFinishedListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("switchAccount", UnityAndroid.GetUnityActivity(), listener);
            });
        }

        /// <summary>
        /// 设置服务器ID
        /// </summary>
        /// <param name="serverId"></param>
        public void SetServer(string serverId)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("setServer", serverId);
            });
        }

        /// <summary>
        /// 注销
        /// </summary>
        public void Logout()
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("logout");
            });
        }

        /// <summary>
        /// 退出游戏
        /// </summary>
        /// <param name="onQuitGameListener"></param>
        public void ShouldQuitGame(OnQuitGameListener onQuitGameListener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("shouldQuitGame", UnityAndroid.GetUnityActivity(), onQuitGameListener);
            });
        }

        /// <summary>
        /// 分享
        /// </summary>
        public void Share()
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("share", UnityAndroid.GetUnityActivity());
            });
        }

        /// <summary>
        /// 游戏评分
        /// </summary>
        public void Comment()
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("comment", UnityAndroid.GetUnityActivity());
            });
        }

        /// <summary>
        /// 获取当前账户信息
        /// </summary>
        /// <returns></returns>
        public User GetCurrentAccount()
        {
            return new User(JavaObject.Call<AndroidJavaObject>("getCurrentAccount"));
        }

        /// <summary>
        /// 检查当前登录状态
        /// </summary>
        /// <returns></returns>
        public bool IsLogin()
        {
            return JavaObject.Call<bool>("isLogin");
        }

        /// <summary>
        /// 充值
        /// </summary>
        /// <param name="money">充值金额，整型，单位元，1 ～ 50000</param>
        /// <param name="mark"> 游戏方订单，支持大小写字母、数字、'|'(竖线)、'-'（中划线）、'_'（下划线），最长32位，不可为空，不可重复</param>
        /// <param name="commodity">商品名称，最长8个字符</param>
        /// <param name="listener"></param>
        public void Recharge(int money, string mark, string commodity, OnRechargeFinishedListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("recharge", UnityAndroid.GetUnityActivity(), money, mark, commodity, listener);
            });
        }

        /// <summary>
        /// 充值
        /// </summary>
        /// <param name="order"></param>
        /// <param name="listener"></param>
        public void Recharge(Order order, OnRechargeFinishedListener listener)
        {
            UnityAndroid.RunOnUiThread(() =>
            {
                JavaObject.Call("recharge", UnityAndroid.GetUnityActivity(), order.GetJavaObject(), listener);
            });
        }

        /// <summary>
        /// 初始化监听器
        /// </summary>
        public abstract class OnInitGlobalListener : AndroidJavaProxy
        {
            public OnInitGlobalListener() : base("cn.m4399.operate.OperateCenter$OnInitGlobalListener")
            {
            }

            internal void onInitFinished(bool isLogin, AndroidJavaObject user)
            {
                OnInitFinished(isLogin, new User(user));
            }

            internal void onUserAccountLogout(bool fromUserCenter)
            {
                OnUserAccountLogout(fromUserCenter);
            }

            internal void onSwitchUserAccountFinished(bool fromUserCenter, AndroidJavaObject user)
            {
                OnSwitchUserAccountFinished(fromUserCenter, new User(user));
            }

            /// <summary>
            /// 初始化完成
            /// </summary>
            /// <param name="isLogin"></param>
            /// <param name="user"></param>
            protected abstract void OnInitFinished(bool isLogin, User user);

            /// <summary>
            /// 用户注销的回调
            /// </summary>
            /// <param name="fromUserCenter"></param>
            protected abstract void OnUserAccountLogout(bool fromUserCenter);


            /// <summary>
            /// 切换用户
            /// </summary>
            protected abstract void OnSwitchUserAccountFinished(bool fromUserCenter, User user);

        }

        /// <summary>
        /// 登录监听器
        /// </summary>
        public abstract class OnLoginFinishedListener : AndroidJavaProxy
        {
            public OnLoginFinishedListener() : base("cn.m4399.operate.OperateCenter$OnLoginFinishedListener")
            {
            }

            internal void onLoginFinished(bool success, int resultCode, AndroidJavaObject user)
            {
                OnLoginFinished(success, resultCode, new User(user));
            }

            /// <summary>
            /// 登录完成
            /// </summary>
            /// <param name="success"></param>
            /// <param name="resultCode"></param>
            /// <param name="user"></param>
            protected abstract void OnLoginFinished(bool success, int resultCode, User user);
        }

        /// <summary>
        /// 退出游戏监听器
        /// </summary>
        public abstract class OnQuitGameListener : AndroidJavaProxy
        {
            public OnQuitGameListener() : base("cn.m4399.operate.OperateCenter$OnQuitGameListener")
            {

            }

            internal void onQuitGame(bool shouldQuit)
            {
                OnQuitGame(shouldQuit);
            }

            /// <summary>
            /// 退出游戏回调
            /// </summary>
            /// <param name="shouldQuit"></param>
            protected abstract void OnQuitGame(bool shouldQuit);
        }

        /// <summary>
        /// 充值监听器
        /// </summary>
        public abstract class OnRechargeFinishedListener : AndroidJavaProxy
        {
            public OnRechargeFinishedListener() : base("cn.m4399.operate.OperateCenter$OnRechargeFinishedListener")
            {

            }

            internal void onRechargeFinished(bool success, int resultCode, string msg)
            {
                OnRechargeFinished(success, resultCode, msg);
            }

            /// <summary>
            /// 通知充值操作完成
            /// </summary>
            /// <param name="success"></param>
            /// <param name="resultCode"></param>
            /// <param name="msg"></param>
            protected abstract void OnRechargeFinished(bool success, int resultCode, string msg);
        }
    }
}
