/*
 * Decompiled with CFR 0.152.
 */
package com.android.volley.toolbox;

import android.os.SystemClock;
import android.text.TextUtils;
import androidx.annotation.VisibleForTesting;
import com.android.volley.Cache;
import com.android.volley.Header;
import com.android.volley.VolleyLog;
import com.android.volley.toolbox.HttpHeaderParser;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class DiskBasedCache
implements Cache {
    private final Map<String, CacheHeader> mEntries = new LinkedHashMap<String, CacheHeader>(16, 0.75f, true);
    private long mTotalSize = 0L;
    private final FileSupplier mRootDirectorySupplier;
    private final int mMaxCacheSizeInBytes;
    private static final int DEFAULT_DISK_USAGE_BYTES = 0x500000;
    @VisibleForTesting
    static final float HYSTERESIS_FACTOR = 0.9f;
    private static final int CACHE_MAGIC = 538247942;

    public DiskBasedCache(final File rootDirectory, int maxCacheSizeInBytes) {
        this.mRootDirectorySupplier = new FileSupplier(){

            @Override
            public File get() {
                return rootDirectory;
            }
        };
        this.mMaxCacheSizeInBytes = maxCacheSizeInBytes;
    }

    public DiskBasedCache(FileSupplier rootDirectorySupplier, int maxCacheSizeInBytes) {
        this.mRootDirectorySupplier = rootDirectorySupplier;
        this.mMaxCacheSizeInBytes = maxCacheSizeInBytes;
    }

    public DiskBasedCache(File rootDirectory) {
        this(rootDirectory, 0x500000);
    }

    public DiskBasedCache(FileSupplier rootDirectorySupplier) {
        this(rootDirectorySupplier, 0x500000);
    }

    @Override
    public synchronized void clear() {
        File[] files = this.mRootDirectorySupplier.get().listFiles();
        if (files != null) {
            for (File file : files) {
                file.delete();
            }
        }
        this.mEntries.clear();
        this.mTotalSize = 0L;
        VolleyLog.d("Cache cleared.", new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public synchronized Cache.Entry get(String key) {
        CacheHeader entry = this.mEntries.get(key);
        if (entry == null) {
            return null;
        }
        File file = this.getFileForKey(key);
        try (CountingInputStream cis = new CountingInputStream(new BufferedInputStream(this.createInputStream(file)), file.length());){
            CacheHeader entryOnDisk = CacheHeader.readHeader(cis);
            if (!TextUtils.equals((CharSequence)key, (CharSequence)entryOnDisk.key)) {
                VolleyLog.d("%s: key=%s, found=%s", file.getAbsolutePath(), key, entryOnDisk.key);
                this.removeEntry(key);
                Cache.Entry entry2 = null;
                return entry2;
            }
            byte[] data = DiskBasedCache.streamToBytes(cis, cis.bytesRemaining());
            Cache.Entry entry3 = entry.toCacheEntry(data);
            return entry3;
        }
        catch (IOException e) {
            VolleyLog.d("%s: %s", file.getAbsolutePath(), e.toString());
            this.remove(key);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void initialize() {
        File rootDirectory = this.mRootDirectorySupplier.get();
        if (!rootDirectory.exists()) {
            if (!rootDirectory.mkdirs()) {
                VolleyLog.e("Unable to create cache dir %s", rootDirectory.getAbsolutePath());
            }
            return;
        }
        File[] files = rootDirectory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            try {
                long entrySize = file.length();
                try (CountingInputStream cis = new CountingInputStream(new BufferedInputStream(this.createInputStream(file)), entrySize);){
                    CacheHeader entry = CacheHeader.readHeader(cis);
                    entry.size = entrySize;
                    this.putEntry(entry.key, entry);
                }
            }
            catch (IOException e) {
                file.delete();
            }
        }
    }

    @Override
    public synchronized void invalidate(String key, boolean fullExpire) {
        Cache.Entry entry = this.get(key);
        if (entry != null) {
            entry.softTtl = 0L;
            if (fullExpire) {
                entry.ttl = 0L;
            }
            this.put(key, entry);
        }
    }

    @Override
    public synchronized void put(String key, Cache.Entry entry) {
        if (this.mTotalSize + (long)entry.data.length > (long)this.mMaxCacheSizeInBytes && (float)entry.data.length > (float)this.mMaxCacheSizeInBytes * 0.9f) {
            return;
        }
        File file = this.getFileForKey(key);
        try {
            BufferedOutputStream fos = new BufferedOutputStream(this.createOutputStream(file));
            CacheHeader e = new CacheHeader(key, entry);
            boolean success = e.writeHeader(fos);
            if (!success) {
                fos.close();
                VolleyLog.d("Failed to write header for %s", file.getAbsolutePath());
                throw new IOException();
            }
            fos.write(entry.data);
            fos.close();
            e.size = file.length();
            this.putEntry(key, e);
            this.pruneIfNeeded();
        }
        catch (IOException e) {
            boolean deleted = file.delete();
            if (!deleted) {
                VolleyLog.d("Could not clean up file %s", file.getAbsolutePath());
            }
            this.initializeIfRootDirectoryDeleted();
        }
    }

    @Override
    public synchronized void remove(String key) {
        boolean deleted = this.getFileForKey(key).delete();
        this.removeEntry(key);
        if (!deleted) {
            VolleyLog.d("Could not delete cache entry for key=%s, filename=%s", key, this.getFilenameForKey(key));
        }
    }

    private String getFilenameForKey(String key) {
        int firstHalfLength = key.length() / 2;
        String localFilename = String.valueOf(key.substring(0, firstHalfLength).hashCode());
        localFilename = localFilename + String.valueOf(key.substring(firstHalfLength).hashCode());
        return localFilename;
    }

    public File getFileForKey(String key) {
        return new File(this.mRootDirectorySupplier.get(), this.getFilenameForKey(key));
    }

    private void initializeIfRootDirectoryDeleted() {
        if (!this.mRootDirectorySupplier.get().exists()) {
            VolleyLog.d("Re-initializing cache after external clearing.", new Object[0]);
            this.mEntries.clear();
            this.mTotalSize = 0L;
            this.initialize();
        }
    }

    private void pruneIfNeeded() {
        if (this.mTotalSize < (long)this.mMaxCacheSizeInBytes) {
            return;
        }
        if (VolleyLog.DEBUG) {
            VolleyLog.v("Pruning old cache entries.", new Object[0]);
        }
        long before = this.mTotalSize;
        int prunedFiles = 0;
        long startTime = SystemClock.elapsedRealtime();
        Iterator<Map.Entry<String, CacheHeader>> iterator = this.mEntries.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, CacheHeader> entry = iterator.next();
            CacheHeader e = entry.getValue();
            boolean deleted = this.getFileForKey(e.key).delete();
            if (deleted) {
                this.mTotalSize -= e.size;
            } else {
                VolleyLog.d("Could not delete cache entry for key=%s, filename=%s", e.key, this.getFilenameForKey(e.key));
            }
            iterator.remove();
            ++prunedFiles;
            if (!((float)this.mTotalSize < (float)this.mMaxCacheSizeInBytes * 0.9f)) continue;
            break;
        }
        if (VolleyLog.DEBUG) {
            VolleyLog.v("pruned %d files, %d bytes, %d ms", prunedFiles, this.mTotalSize - before, SystemClock.elapsedRealtime() - startTime);
        }
    }

    private void putEntry(String key, CacheHeader entry) {
        if (!this.mEntries.containsKey(key)) {
            this.mTotalSize += entry.size;
        } else {
            CacheHeader oldEntry = this.mEntries.get(key);
            this.mTotalSize += entry.size - oldEntry.size;
        }
        this.mEntries.put(key, entry);
    }

    private void removeEntry(String key) {
        CacheHeader removed = this.mEntries.remove(key);
        if (removed != null) {
            this.mTotalSize -= removed.size;
        }
    }

    @VisibleForTesting
    static byte[] streamToBytes(CountingInputStream cis, long length) throws IOException {
        long maxLength = cis.bytesRemaining();
        if (length < 0L || length > maxLength || (long)((int)length) != length) {
            throw new IOException("streamToBytes length=" + length + ", maxLength=" + maxLength);
        }
        byte[] bytes = new byte[(int)length];
        new DataInputStream(cis).readFully(bytes);
        return bytes;
    }

    @VisibleForTesting
    InputStream createInputStream(File file) throws FileNotFoundException {
        return new FileInputStream(file);
    }

    @VisibleForTesting
    OutputStream createOutputStream(File file) throws FileNotFoundException {
        return new FileOutputStream(file);
    }

    private static int read(InputStream is) throws IOException {
        int b = is.read();
        if (b == -1) {
            throw new EOFException();
        }
        return b;
    }

    static void writeInt(OutputStream os, int n) throws IOException {
        os.write(n >> 0 & 0xFF);
        os.write(n >> 8 & 0xFF);
        os.write(n >> 16 & 0xFF);
        os.write(n >> 24 & 0xFF);
    }

    static int readInt(InputStream is) throws IOException {
        int n = 0;
        n |= DiskBasedCache.read(is) << 0;
        n |= DiskBasedCache.read(is) << 8;
        n |= DiskBasedCache.read(is) << 16;
        return n |= DiskBasedCache.read(is) << 24;
    }

    static void writeLong(OutputStream os, long n) throws IOException {
        os.write((byte)(n >>> 0));
        os.write((byte)(n >>> 8));
        os.write((byte)(n >>> 16));
        os.write((byte)(n >>> 24));
        os.write((byte)(n >>> 32));
        os.write((byte)(n >>> 40));
        os.write((byte)(n >>> 48));
        os.write((byte)(n >>> 56));
    }

    static long readLong(InputStream is) throws IOException {
        long n = 0L;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 0;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 8;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 16;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 24;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 32;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 40;
        n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 48;
        return n |= ((long)DiskBasedCache.read(is) & 0xFFL) << 56;
    }

    static void writeString(OutputStream os, String s) throws IOException {
        byte[] b = s.getBytes("UTF-8");
        DiskBasedCache.writeLong(os, b.length);
        os.write(b, 0, b.length);
    }

    static String readString(CountingInputStream cis) throws IOException {
        long n = DiskBasedCache.readLong(cis);
        byte[] b = DiskBasedCache.streamToBytes(cis, n);
        return new String(b, "UTF-8");
    }

    static void writeHeaderList(List<Header> headers, OutputStream os) throws IOException {
        if (headers != null) {
            DiskBasedCache.writeInt(os, headers.size());
            for (Header header : headers) {
                DiskBasedCache.writeString(os, header.getName());
                DiskBasedCache.writeString(os, header.getValue());
            }
        } else {
            DiskBasedCache.writeInt(os, 0);
        }
    }

    static List<Header> readHeaderList(CountingInputStream cis) throws IOException {
        int size = DiskBasedCache.readInt(cis);
        if (size < 0) {
            throw new IOException("readHeaderList size=" + size);
        }
        ArrayList<Header> result = size == 0 ? Collections.emptyList() : new ArrayList<Header>();
        for (int i = 0; i < size; ++i) {
            String name = DiskBasedCache.readString(cis).intern();
            String value = DiskBasedCache.readString(cis).intern();
            result.add(new Header(name, value));
        }
        return result;
    }

    @VisibleForTesting
    static class CountingInputStream
    extends FilterInputStream {
        private final long length;
        private long bytesRead;

        CountingInputStream(InputStream in, long length) {
            super(in);
            this.length = length;
        }

        @Override
        public int read() throws IOException {
            int result = super.read();
            if (result != -1) {
                ++this.bytesRead;
            }
            return result;
        }

        @Override
        public int read(byte[] buffer, int offset, int count) throws IOException {
            int result = super.read(buffer, offset, count);
            if (result != -1) {
                this.bytesRead += (long)result;
            }
            return result;
        }

        @VisibleForTesting
        long bytesRead() {
            return this.bytesRead;
        }

        long bytesRemaining() {
            return this.length - this.bytesRead;
        }
    }

    @VisibleForTesting
    static class CacheHeader {
        long size;
        final String key;
        final String etag;
        final long serverDate;
        final long lastModified;
        final long ttl;
        final long softTtl;
        final List<Header> allResponseHeaders;

        private CacheHeader(String key, String etag, long serverDate, long lastModified, long ttl, long softTtl, List<Header> allResponseHeaders) {
            this.key = key;
            this.etag = "".equals(etag) ? null : etag;
            this.serverDate = serverDate;
            this.lastModified = lastModified;
            this.ttl = ttl;
            this.softTtl = softTtl;
            this.allResponseHeaders = allResponseHeaders;
        }

        CacheHeader(String key, Cache.Entry entry) {
            this(key, entry.etag, entry.serverDate, entry.lastModified, entry.ttl, entry.softTtl, CacheHeader.getAllResponseHeaders(entry));
        }

        private static List<Header> getAllResponseHeaders(Cache.Entry entry) {
            if (entry.allResponseHeaders != null) {
                return entry.allResponseHeaders;
            }
            return HttpHeaderParser.toAllHeaderList(entry.responseHeaders);
        }

        static CacheHeader readHeader(CountingInputStream is) throws IOException {
            int magic = DiskBasedCache.readInt(is);
            if (magic != 538247942) {
                throw new IOException();
            }
            String key = DiskBasedCache.readString(is);
            String etag = DiskBasedCache.readString(is);
            long serverDate = DiskBasedCache.readLong(is);
            long lastModified = DiskBasedCache.readLong(is);
            long ttl = DiskBasedCache.readLong(is);
            long softTtl = DiskBasedCache.readLong(is);
            List<Header> allResponseHeaders = DiskBasedCache.readHeaderList(is);
            return new CacheHeader(key, etag, serverDate, lastModified, ttl, softTtl, allResponseHeaders);
        }

        Cache.Entry toCacheEntry(byte[] data) {
            Cache.Entry e = new Cache.Entry();
            e.data = data;
            e.etag = this.etag;
            e.serverDate = this.serverDate;
            e.lastModified = this.lastModified;
            e.ttl = this.ttl;
            e.softTtl = this.softTtl;
            e.responseHeaders = HttpHeaderParser.toHeaderMap(this.allResponseHeaders);
            e.allResponseHeaders = Collections.unmodifiableList(this.allResponseHeaders);
            return e;
        }

        boolean writeHeader(OutputStream os) {
            try {
                DiskBasedCache.writeInt(os, 538247942);
                DiskBasedCache.writeString(os, this.key);
                DiskBasedCache.writeString(os, this.etag == null ? "" : this.etag);
                DiskBasedCache.writeLong(os, this.serverDate);
                DiskBasedCache.writeLong(os, this.lastModified);
                DiskBasedCache.writeLong(os, this.ttl);
                DiskBasedCache.writeLong(os, this.softTtl);
                DiskBasedCache.writeHeaderList(this.allResponseHeaders, os);
                os.flush();
                return true;
            }
            catch (IOException e) {
                VolleyLog.d("%s", e.toString());
                return false;
            }
        }
    }

    public static interface FileSupplier {
        public File get();
    }
}

